<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\PrepaidCard;
use App\Models\Vehicle;
use App\Models\Payment;
use App\Models\PrepaidSummary;
use App\Models\Fare;
use Illuminate\Support\Facades\DB;
use App\Traits\PaginatorHelper;
use Barryvdh\Snappy\Facades\SnappyPdf;
use App\Jobs\SendRechargePrepaidSmsJob;

class PrepaidCardController extends Controller
{
    use PaginatorHelper;

    public function index(Request $request)
    {
        $helper       = $this->processRequest($request);
        $prepaidCards = PrepaidCard::whereNotNull('id');
        if ($request->profileId) {
            $prepaidCards->where('profile_id', $request->profileId);
        }
        $prepaidCards = $this->applyFilters($prepaidCards, $helper,
            ['code', 'profile.names,surnames,id_card']);

        return response()->json($prepaidCards);
    }

    public function search(Request $request)
    {
        // Recover the number gived by the user
        $code = $request->number;
        $password = $request->password;
        $plate = $request->plate;

        if (!$code || !$plate) {
            return response()->json([
                "status" => 0,
                "message" => "Informacion incompleta",
            ]);
        }

        // Verify if the number is a Prepaid Card code or not
        // Will be a Prepaid Card code if the number gived have only numbers
        $isPrepaidCardCode = (is_numeric($code)) ? true : false;

        try {
            // fill with zeros to the left
            $code = str_pad($code, 10, "0", STR_PAD_LEFT);
            $prepaidCard = PrepaidCard::where('code', $code)->first();

            // If the card was not recover
            if (!$prepaidCard) {
                return response()->json([
                    "status" => 0,
                    "message" => "Carnet o Contrasena incorrecta",
                ]);
            }

            // If prepaid card is disabled
            if (!$prepaidCard->status) {
                return response()->json([
                    "status" => 0,
                    "message" => "Carnet deshabilitado",
                ]);
            }

            // If the password is incorrect or is not direct
            if (($prepaidCard->password != $password) && (!$prepaidCard->direct)) {
                return response()->json([
                    "status" => 0,
                    "message" => "Carnet o Contrasena incorrecta",
                ]);
            }

            // Recover the vehicles and search the plate
            $vehicle = $prepaidCard->vehicles()->where('plate', $plate)->first();
            // if the plate is not asociated to the card
            if (!$vehicle) {
                return response()->json([
                    "status" => 0,
                    "message" => "Placa no registrada o no asociado a el carnet",
                ]);
            }

            // Recover the fare asociated to the vehicle
            $fare = Fare::where('vehicle_type_id', $vehicle->vehicle_type_id)->first();
            // If the fare is not found
            if (!$fare) {
                return response()->json([
                    "status" => 0,
                    "message" => "Tarifa asociada no registrada",
                ]);
            }

            // If the balance is insuficient and the prepaid card is not exonerated
            if (($prepaidCard->balance < $fare->value) && (!$prepaidCard->exonerated)) {
                return response()->json([
                    "status" => 0,
                    "message" => "Saldo Insuficiente: " . $prepaidCard->balance,
                ]);
            }

            $vehicle->vehicleType;
            $vehicle->vehicleBrand;

            $data = new PrepaidCard();
            $data->cardId = $prepaidCard->id;
            $data->exonerated = $prepaidCard->exonerated;
            $data->direct = $prepaidCard->direct;
            $data->status = $prepaidCard->status;
            $data->balance = $prepaidCard->balance;
            $data->vehicle = $vehicle;

            return response()->json([
                "status" => 1,
                "message" => "Informacion Validad exitosamente",
                "data" => $data,
            ]);
        } catch(\Exception $e) {
            \Log::debug($e);
            return response()->json([
                "status" => 0,
                "message" => "Ocurrio un error procesando el carnet. Pongase en contacto con el administrador",
            ]);
        } catch(\Error $e) {
            \Log::debug($e);
            return response()->json([
                "status" => 0,
                "message" => "Ocurrio un error procesando el carnet. Pongase en contacto con el administrador",
            ]);
        }
    }

    public function validatePassword(Request $request)
    {
        $idCard = $request->idCard;
        $password = $request->password;
        $card = PrepaidCard::where('id', $idCard)->where('password', $password)->first();
        if ($card) {
            return response()->json([
                "status" => 1,
                "message" => "Clave validada Exitosamente"
            ]);
        } else {
            return response()->json([
                "status" => 0,
                "message" => "Clave Invalida"
            ]);
        }
    }

    private function searchVehicle($plate)
    {
        $vehicle = Vehicle::with(['profile', 'vehicleType', 'prepaidCards'])
                            ->where('plate', $plate)
                            ->get();
        return $vehicle  ?? null;
    }

    public function generatePrepaidCard(Request $request)
    {
        $profileId = $request->profileId;
        $code = $this->generatePrepaidCardCode();
        try{
            $newPrepaidCard = PrepaidCard::create([
                "profile_id" => $profileId,
                "code" => $code,
                "password" => substr($code, 6),
                "balance" => 0,
                "exonerated" => 0,
                "direct" => 0,
                "status" => 1,
            ]);
            return response()->json([
                "status" => 1,
                "item" => $newPrepaidCard,
                "message" => "Carnet ". $code . " generado exitosamente"
            ]);
        } catch(\Exception $e) {
            // Return Response
            return response()->json([
                "status" => 0,
                "message" => $e
            ]);
        }
    }

    public function changePassword(Request $request)
    {
        $code = $request->code;
        $card = PrepaidCard::where('code', $code)->first();
        if(!$card){
            return response()->json([
                "status" => 0,
                "message" => "Carnet no encontrado"
            ]);
        }

        $password = $request->password;
        if(strlen($password) != 4 || !is_numeric($password)) {
            return response()->json([
                "status" => 0,
                "message" => "Contrasena Invalida, debe tener 4 digitos"
            ]);
        }

        $card->password = $password;
        $card->save();

        return response()->json([
            "status" => 1,
            "message" => "Contrasena cambiada exitosamente"
        ]);
    }

    public function makeRecharge(Request $request)
    {
        $cardId = $request->cardId;
        $ammount = $request->ammount;
        $lastDigits = $request->lastDigits;
        $paymentTypeId = $request->paymentTypeId;

        if ($lastDigits) {
            $recharge = Payment::where('last_digits', $lastDigits)->first();
            if ($recharge) {
                return response()->json([
                    "status" => 0,
                    'message' => "El codigo de referencia ya se encuentra asociado a un pago"
                ]);
            }
        }

        DB::beginTransaction();
        try{
            $card = PrepaidCard::find($cardId);
            if (!$card){
                return response()->json([
                    "status" => 0,
                    'message' => "Tarjeta Prepago no encontrada en el sistema"
                ]);
            }

            $card->balance += floatval($ammount);
            $card->save();

            $payment = Payment::create([
                "last_digits" => $lastDigits,
                "payment_method_id" => $paymentTypeId,
                "amount" => $ammount,
            ]);

            $summary = PrepaidSummary::create([
                "prepaid_card_id" => $cardId,
                "type" => "add",
                "code" => $this->generatePrepaidCardCode(),
                "amount" => $ammount,
                "concept" => "Recarga realizada en oficina",
                "new_balance" => $card->balance,
            ]);

            if (!$payment || !$summary){
                return response()->json([
                    "status" => 0,
                    'message' => "No se pudo registrar la recarga"
                ]);
            }

            $payment->prepaidSummaries()->attach($summary);

            DB::commit();
            // Send SMS to the client
            SendRechargePrepaidSmsJob::dispatch($card->profile, $summary);
            return response()->json([
                "status" => 1,
                'message' => "Recarga realizada exitosamente, nuevo saldo: "
            ]);

        } catch(Exception $e){
            DB::rollback();
            return response()->json([
                "status" => 0,
                'message' => "Ocurrio un error, intente mas tarde"
            ]);
        }

    }

    public function delete(Request $request)
    {
        $id = $request->id;
        try{
            $card = PrepaidCard::find($id);
            if(!$card){
                return response()->json([
                    "status" => 0,
                    "message" => "Carnet No existe en la base de datos"
                ]);
            }

            if ($card->prepaidSummaries->count() > 0) {
                return response()->json([
                    "status" => 0,
                    "message" => "Este Carnet ya tiene movimientos registrados en la base de datos, es imposible eliminarlo"
                ]);
            }

            $card->delete();
            return response()->json([
                "status" => 1,
                "message" => "Carnet Prepagado ". $card->names . " eliminado exitosamente"
            ]);

        } catch (\Exception $e) {
            return response()->json([
                "status" => 0,
                "message" => "Ocurrio un error, intente mas tarde, si el problema persiste comuniquese con su administrador"
            ]);
        }
    }

    public function printPrepaidCard(Request $request)
    {
        $profile = $request->user()->profile;
        $id = $request->id;
        $card = PrepaidCard::with('profile.documentType')->find($id);
        if (!$card) {
            return response()->json([
                "status" => 0,
                "message" => "Carnet no encontrado"
            ], 404);
        }

        if ($profile->profile_type_id == 3){
            return response()->json([
                    "status" => 0,
                    "message" => "Error de Autenticacion"
            ], 403);
        }

        if ($profile->profile_type_id == 2){
            if ($card->profile_id != $profile->id){
                return response()->json([
                    "status" => 0,
                    "message" => "Error de Autenticacion"
                ]);
            }
        }

        $pdf = SnappyPDF::loadView('pdfcard', compact('card'));
        $pdf->setOption('margin-bottom', 0);
        $pdf->setOption('margin-top', 0);
        $pdf->setOption('margin-left', 0);
        $pdf->setOption('margin-right', 0);
        $pdf->setOption('page-size', 'Letter');
        $pdf->setOption('lowquality', false);
        $pdf->setOption('disable-smart-shrinking', true);
        $pdf->setOption('images', true);
        $pdf->setOption('orientation', 'Portrait');
        $pdf->setOption('dpi', 300);
        $pdf->setOption('image-quality', 100);
        return $pdf->inline();
    }

    public function toggleDirect(Request $request)
    {
        $id = $request->id;
        try{
            $card = PrepaidCard::find($id);
            if(!$card){
                return response()->json([
                    "status" => 0,
                    "message" => "Carnet No existe en la base de datos"
                ]);
            }

            $card->direct = !$card->direct;
            $card->save();
            return response()->json([
                "status" => 1,
                "message" => "Estatus directo del carnet " . $card->code . " ha sido cambiado exitosamente"
            ]);

        } catch (\Exception $e) {
            return response()->json([
                "status" => 0,
                "message" => "Ocurrio un error, intente mas tarde, si el problema persiste comuniquese con su administrador"
            ]);
        }
    }

    public function toggleStatus(Request $request)
    {
        $id = $request->id;
        try{
            $card = PrepaidCard::find($id);
            if(!$card){
                return response()->json([
                    "status" => 0,
                    "message" => "Carnet No existe en la base de datos"
                ]);
            }

            $card->status = !$card->status;
            $card->save();
            return response()->json([
                "status" => 1,
                "message" => "Estatus del Carnet " . $card->code . " ha sido cambiado exitosamente"
            ]);

        } catch (\Exception $e) {
            return response()->json([
                "status" => 0,
                "message" => "Ocurrio un error, intente mas tarde, si el problema persiste comuniquese con su administrador"
            ]);
        }
    }

    public function updateVehicle(Request $request)
    {
        $cardId = $request->cardId;
        try{
            $card = PrepaidCard::find($cardId);
            if(!$card){
                return response()->json([
                    "status" => 0,
                    "message" => "Carnet No existe en la base de datos"
                ]);
            }

            $vehicleId = $request->vehicleId;
            $vehicle = $card->vehicles()->where('vehicle_id', $vehicleId)->first();
            if ($vehicle) {
                $card->vehicles()->detach($vehicleId);
            } else {
                $card->vehicles()->attach($vehicleId);
            }
            return response()->json([
                "status" => 1,
                "message" => "Operacion realizada exitosamente"
            ]);

        } catch (\Exception $e) {
            return response()->json([
                "status" => 0,
                "error" => $e->getMessage(),
                "message" => "Ocurrio un error, intente mas tarde, si el problema persiste comuniquese con su administrador"
            ]);
        }
    }

    public function getVehicles(Request $request)
    {
        $id = $request->id;
        try{
            $card = PrepaidCard::find($id);
            if(!$card){
                return response()->json([
                    "status" => 0,
                    "message" => "Carnet No existe en la base de datos"
                ]);
            }

            if(!$card->status){
                return response()->json([
                    "status" => 0,
                    "message" => "El Carnet se encuentra deshabilitado"
                ]);
            }

            $vehicles = $card->vehicles;

            return response()->json($vehicles);

        } catch (\Exception $e) {
            return response()->json([
                "status" => 0,
                "message" => "Ocurrio un error, intente mas tarde, si el problema persiste comuniquese con su administrador"
            ]);
        }
    }

    private function generatePrepaidCardCode()
    {
        $randomNumber = random_int(100000, 999999);
        $newCode = "0000" . strval($randomNumber);
        $card = PrepaidCard::where('code', $newCode)->first();
        if ($card) {
            $this->generatePrepaidCardCode();
        } else {
            return $newCode;
        }
    }

}
