<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use App\Models\Profile;
use App\Models\User;
use App\Models\Vehicle;
use App\Models\PrepaidCard;

class ProfileController extends Controller
{
    public function index(Request $request)
    {
        $rows = $request->size ?? 10;
        $search = $request->search ?? null;
        $profiles = Profile::with(['documentType', 'profileType']);
        if ($search) {
            $profiles->where("id_card", "LIKE", "%".$search."%");
            $profiles->orWhere("names", "LIKE", "%".$search."%");
            $profiles->orWhere("surnames", "LIKE", "%".$search."%");
        }
        $profiles = $profiles->paginate($rows);

        return response()->json($profiles);
    }

    public function search(Request $request)
    {
        $documentType = $request->documentType;
        $idCard = $request->idCard;
        $profile = Profile::with(['user', 'vehicles', 'documentType'])
                        ->where('document_type_id', $documentType)
                        ->where('id_card', $idCard)
                        ->first();

        if ($profile) {
            return response()->json([
                "status" => 1,
                "item" => $profile,
                "message" => "Cliente encontrado"
            ]);
        } else {
            return response()->json([
                "status" => 0,
                "message" => "Cliente No encontrado"
            ]);
        }
    }

    public function save(Request $request)
    {
        $id = $request->id;
        try{
            // Begin of the transaction
            DB::beginTransaction();
            if ($id) {
                $profile = Profile::with('user')->find($id);
                $profile->user->email = $request->email;

                $user = $profile->user;
            } else {
                $profile = new Profile();

                $user = User::create([
                    "email" => $request->email,
                    "password" => "$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/ig",
                    "email_verified_at" => now()
                ]);
                $profile->user_id = $user->id;
            }

            // Verification
            $verify = Profile::where("id_card", $request->idCard)
                                ->where("document_type_id", $request->documentType)
                                ->first();
            if ($verify && $verify->id != $profile->id) {
                DB::rollback();
                return response()->json([
                    "status" => 0,
                    "message" => "Documento de Identidad ya le pertenece a otro usuario: " . $verify->names . " " . $verify->surnames ?? ""
                ]);
            }

            $profile->id_card = $request->idCard;
            $profile->document_type_id = $request->documentType;
            $profile->profile_type_id = $request->profileTypeId;
            $profile->names = $request->names;
            $profile->surnames = $request->surnames;
            $profile->phone = $request->phone;
            $profile->address = $request->address;

            $profile->save();
            $profile->documentType;
            $profile->vehicles;
            $profile->user;

            $password = $request->password;
            if($password && strlen($password) == 4) {
                $user->password = Hash::make($password);
                $user->save();
            }

            // Write in the database
            DB::commit();
            if ($profile) {
                return response()->json([
                    "status" => 1,
                    "item" => $profile,
                    "message" => "Usuario actualizado exitosamente"
                ]);
            } else {
                return response()->json([
                    "status" => 0,
                    "message" => "Cliente No encontrado"
                ]);
            }
        } catch(\Exception $e) {
            // Make a Rollback
            DB::rollback();

            // Return Response
            return response()->json([
                "status" => 0,
                "message" => $e->errorInfo
            ]);
        }
    }

    public function getVehicles(Request $request)
    {
        $profileId = $request->profileId;
        $vehicles = Vehicle::where('profile_id', $profileId)->get();
        return response()->json($vehicles);
    }

    public function getPrepaidCards(Request $request)
    {
        $profileId = $request->profileId;
        $vehicles = PrepaidCard::select(['id', 'code', 'password', 'balance'])->where('profile_id', $profileId)->get();
        return response()->json($vehicles);
    }

    public function delete(Request $request)
    {
        $id = $request->id;
        try{
            $profile = Profile::find($id);
            if ($profile->vehicles->count() > 0
                || $profile->prepaidCards->count() > 0
                || $profile->turns->count() > 0) {
                return response()->json([
                    "status" => 0,
                    "message" => "Este Usuario / Cliente ya tiene movimientos registrados en la base de datos, es imposible eliminarlo"
                ]);
            }

            $profile->delete();
            return response()->json([
                "status" => 1,
                "message" => "Usuario ". $profile->names . " ". $profile->surnames . " eliminado exitosamente"
            ]);

        } catch (\Exception $e) {
            return response()->json([
                "status" => 0,
                "message" => "Ocurrio un error, intente mas tarde, si el problema persiste comuniquese con su administrador"
            ]);
        }
    }

    public function workers(Request $request)
    {
        $rows = $request->size ?? 10;
        $search = $request->search ?? null;
        $profiles = Profile::with(['documentType', 'profileType']);
        $profiles->where('profile_type_id', '>', 2);
        if ($search) {
            $profiles->where("id_card", "LIKE", "%".$search."%");
            $profiles->orWhere("names", "LIKE", "%".$search."%");
            $profiles->orWhere("surnames", "LIKE", "%".$search."%");
        }
        $profiles = $profiles->paginate($rows);

        return response()->json($profiles);
    }
}
